using System;
using System.Drawing;
using System.Collections;
using System.Reflection;

using Microsoft.DirectX;

using DarkStrideToolbox;

namespace DarkStride.StellarLanes.SharedDLL
{
	public class XMissile : Entity
	{
		#region Properties
		private double m_nExplosionStartTime = 0;
		private bool m_bExploding = false;
		private string m_sThrustPMGUID = "";
		private Vector2 m_vSize = new Vector2( 4,16 );

		private bool m_bHavePlayedExplodingSound = false;
		private bool m_bHavePlayedLaunchSound = false;

		private double m_nSecondsTillActivation = .5;
		private double m_nSecondsTillDeActivation = 5;

		private double m_nImpactDamage = 350;
		private double m_nExplodeDamage = 150;
		#endregion


		public XMissile()
		{
			base.MaxVelocity = 0;
			base.EntityType = enumEntityType.Projectile;
		}
        public override void Advance(Session oSession, double nElapsedTime)
		{
            double nAccel = 260;


            base.Advance(oSession, nElapsedTime);

			if( this.IsDead == false )
			{
				if( m_bExploding == true && m_bHavePlayedExplodingSound == false && Globals.Inst().IAmTheServer == false )
				{
					m_bHavePlayedExplodingSound = true;
                    Globals.PlaySound("XMissile_Explosion", this.Location);
				}
				if( m_bHavePlayedLaunchSound == false && Globals.Inst().IAmTheServer == false )
				{
					m_bHavePlayedLaunchSound = true;
                    Globals.PlaySound("XMissile_Launch", this.Location);
				}
				if( m_bExploding == true && m_nExplosionStartTime == 0 )
				{
					m_nExplosionStartTime = base.TotalElapsedTime;
				}
				if( base.TotalElapsedTime > m_nSecondsTillActivation )
				{
					base.RecoilsAfterCollision = false;
				}
			
				if( base.TotalElapsedTime > m_nSecondsTillActivation && 
					base.TotalElapsedTime < m_nSecondsTillDeActivation && 
					m_bExploding == false )
				{
					base.Vel = new Vector2( (float)( base.Vel.X + Math.Cos( base.Angle - Math.PI / 2.0 ) * nAccel * nElapsedTime ),
                                            (float)(base.Vel.Y + Math.Sin(base.Angle - Math.PI / 2.0) * nAccel * nElapsedTime ) );
					//Set our emmiter position
					UpdatePMData();
				}
				else if( Globals.Inst().IAmTheServer == false )
				{
					ShutOffPM();
				}
			}
			else
			{
				ShutOffPM();
			}


			if( 
				(
					( base.TotalElapsedTime > 7 && base.IsDead == false )
					||
					( base.TotalElapsedTime > 40 && base.IsDead == true )
				)
				||
				( m_bExploding == true && GetExplosionFrame() > 9 ) )
			{
                oSession.RemoveEntity(this);
			}
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			string sTextureKey = "";
			LoadedTexture oLoadedTexture = null;
			Vector2 vScreenPt = Vector2.Empty;
			Vector2 vWorldUpperLeftCorner = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor)
			{
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,base.Pos );

				//Draw our ship
				if( m_bExploding == true && GetExplosionFrame() <= 9 )
				{
					sTextureKey = "XMissile_Explo_" + GetExplosionFrame().ToString();

					oLoadedTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sTextureKey );
					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - oLoadedTexture.Size.X ),
						(int)( vScreenPt.Y - oLoadedTexture.Size.Y ),
						(int)( oLoadedTexture.Size.X * 2 * oRenderSettings.ZoomLevel ),
						(int)( oLoadedTexture.Size.Y * 2 * oRenderSettings.ZoomLevel ) );
					Globals.Inst().GameEngine.RenderTexture2D( sTextureKey,
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( 0,0 ),0,oRenderSettings.PercentTransparent,
						false,oRenderSettings.BaseDrawColor.ToArgb() );
				}
				else
				{
					vWorldUpperLeftCorner = new Vector2((float)( this.Pos.X - m_vSize.X / 2.0 ),
						                                (float)( this.Pos.Y - m_vSize.Y / 2.0 ) );
					vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldUpperLeftCorner );

					oRenderRect = new System.Drawing.Rectangle( 
						(int)vScreenPt.X,(int)vScreenPt.Y,
						(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
						(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );

					Globals.Inst().GameEngine.RenderTexture2D( "XMissile_Projectile",
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( oRenderRect.Width / 2.0f,oRenderRect.Height / 2.0f ),
						base.Angle/* + Math.PI / 2.0*/,
						oRenderSettings.PercentTransparent,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
			}
		}

		private void UpdatePMData()
		{
            Vector2 vEmitPoint = Vector2.Empty;
            Vector2 vRotatedWorldPos = Vector2.Empty;
			DSParticleManager oPMan = null;
			double nAngle = 0;

			if( Globals.Inst().IAmTheServer == false )
			{
				if( m_sThrustPMGUID.Length > 0 )
				{
					oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
				}
				if( oPMan == null )
				{
					oPMan = SLEntityEffect.StartMissileThrust( new Vector2( 0,0 ),new Vector2( 0,0 ),this.Location.ZoneID.ToString() );
					m_sThrustPMGUID = oPMan.GUID;
				}
				nAngle = base.Angle + Math.PI;
                vEmitPoint = new Vector2(this.Pos.X, (float)(this.Pos.Y + m_vSize.Y / 2.0));
                vRotatedWorldPos = GetRotatedWorldPt(vEmitPoint);

                nAngle = this.Angle + Math.PI / 2.0;
                oPMan.EmitterPosition = new Vector2(vRotatedWorldPos.X, vRotatedWorldPos.Y);
				oPMan.EmitterDirection = new Vector2( (float)Math.Cos( nAngle ),(float)Math.Sin( nAngle ) );
				//oPMan.EmittersVelocity = new Vector2( base.Vel.X,base.Vel.Y );

                oPMan.EmitVelocity = 50 - 100.0f * (this.Vel.Length() / 500.0f);
                oPMan.EmittersVelocity = Vector2.Empty;
			}
		}
		private void ShutOffPM()
		{
			DSParticleManager oPMan = null;

			if( m_sThrustPMGUID.Length > 0 )
			{
				oPMan = Globals.Inst().GameEngine.ParticleSystem.GetParticleManager( m_sThrustPMGUID );
				if( oPMan != null )
				{
					oPMan.RunDown = true;
				}

				if( m_bExploding == true )
				{
					oPMan.ParticleLifeTimeInSeconds /= 3.0;
				}

				m_sThrustPMGUID = "";
			}
		}

        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			if( this.IsDead == false && oCollidedWith.OwnerSocketID != this.OwnerSocketID && oCollidedWith.EntityType != enumEntityType.Loot )
			{
				if( m_bExploding == true )
				{
					oCollidedWith.DealDamage( oSession,this,nElapsedTime * m_nExplodeDamage,oCollisionRegion );
				}
				else if( this.TotalElapsedTime > m_nSecondsTillActivation )
				{
					this.RecoilsAfterCollision = false;
					m_bExploding = true;
					m_nExplosionStartTime = base.TotalElapsedTime;
                    oCollidedWith.DealDamage(oSession, this, m_nImpactDamage, oCollisionRegion);
					base.Vel = new Vector2( 0,0 );
				}
				else
				{
					this.AngularMomentum = DSMisc.GetRnd() - .5;
					this.IsDead = true;
				}
			}

			return( true );
		}
        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
            base.DealDamage(oSession, oEntityDealingDamage, nAmount, oDamageWhere);

			if( m_bExploding == false )
			{
				ShutOffPM();

				this.IsDead = false;
				this.RecoilsAfterCollision = false;
				m_bExploding = true;
				m_nExplosionStartTime = base.TotalElapsedTime;
				base.Vel = new Vector2( 0,0 );
			}

			return( 0 );
		}
		public override ArrayList GetRegions()
		{
			RegionCircle oCircle = null;
			ArrayList oRegions = new ArrayList();

			if( m_bExploding == true )
			{
				//Increase our damage radius if we are xploding
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y );
			}
			else
			{
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y / 2.0f );
			}

			//if( base.TotalElapsedTime > m_nSecondsTillActivation )
			//{
			oRegions.Add( oCircle );
			//}

			return( oRegions );
		}
		private int GetExplosionFrame()
		{
			double nTotalExploTime = base.TotalElapsedTime - m_nExplosionStartTime;
			int nFrame = (int)Math.Floor( nTotalExploTime / .03 );
			if( nFrame < 0 )
			{
				nFrame = 0;
			}

			return( nFrame );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nExplosionStartTime );
			oSerialize.Set( 2,m_bExploding );
			oSerialize.Set( 3,m_nImpactDamage );
			oSerialize.Set( 4,m_nExplodeDamage );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			//m_nExplosionStartTime = oSerialize.GetDouble( 1 );
			m_bExploding = oSerialize.GetBool( 2 );
			m_nImpactDamage = oSerialize.GetDouble( 3 );
			m_nExplodeDamage = oSerialize.GetDouble( 4 );
		}


		#region Properties
		public string ThrustPMGUID
		{
			get
			{
				return( m_sThrustPMGUID );
			}
			set
			{
				m_sThrustPMGUID = value;
			}
		}
		public override long EXPValue
		{
			get
			{
				return( 0 );
			}
		}
		public double ImpactDamage
		{
			get
			{
				return( m_nImpactDamage );
			}
			set
			{
				m_nImpactDamage = value;
			}
		}
		public double ExplodeDamage
		{
			get
			{
				return( m_nExplodeDamage );
			}
			set
			{
				m_nExplodeDamage = value;
			}
		}
		#endregion
	}

	public class UrquanPlasmaProjectile : Entity
	{
		#region Properties
		private double m_nImpactDamage = 40;
		private double m_nExplodeDamage = 5;

		private double m_nExplosionStartTime = 0;
		private bool m_bExploding = false;
		private bool m_bHavePlayedExplodingSound = false;
		private bool m_bHavePlayedLaunchSound = false;
		private Vector2 m_vSize = new Vector2( GraphicConstants.m_cGRID_WIDTH,2 * GraphicConstants.m_cGRID_HEIGHT );
		#endregion


		public UrquanPlasmaProjectile()
		{
			base.RecoilsAfterCollision = false;
			base.MaxVelocity = 0;
			base.EntityType = enumEntityType.Projectile;
		}


        public override void Advance(Session oSession, double nElapsedTime)
		{
			//double nAccel = 80;


			if( m_bExploding == true && m_nExplosionStartTime == 0 )
			{
				m_nExplosionStartTime = base.TotalElapsedTime;
			}
			if( m_bExploding == true && m_bHavePlayedExplodingSound == false && Globals.Inst().IAmTheServer == false )
			{
				m_bHavePlayedExplodingSound = true;
                Globals.PlaySound("Urquan_Plasma_Explosion", this.Location);
			}
			if( m_bHavePlayedLaunchSound == false && Globals.Inst().IAmTheServer == false )
			{
				m_bHavePlayedLaunchSound = true;
                Globals.PlaySound("Urquan_Plasma_Launch", this.Location);
			}

			if( base.TotalElapsedTime > 3 )
			{
                oSession.RemoveEntity(this);
			}

			if( m_bExploding == false )
			{
                base.Advance(oSession, nElapsedTime);
			}
			else
			{
				//Advance our explosion
				base.TimeSinceLastLiteSend += nElapsedTime;
				base.TotalElapsedTime += nElapsedTime;
				base.TimeSinceLastCollision += nElapsedTime;
				base.TimeSinceLastFullSend += nElapsedTime;
			}

			if( m_bExploding == true && GetExplosionFrame() > 19 )
			{
                oSession.RemoveEntity(this);
			}
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			double nAngle = 0;
			string sTextureKey = "";
			LoadedTexture oLoadedTexture = null;
			Vector2 vScreenPt = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,base.Pos );

				//Draw our ship
				if( m_bExploding == true && GetExplosionFrame() <= 19 )
				{
					sTextureKey = "Plasma_Explo_" + GetExplosionFrame().ToString();

					oLoadedTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sTextureKey );
					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - oLoadedTexture.Size.X / 2f ),
						(int)( vScreenPt.Y - oLoadedTexture.Size.Y / 2f ),
						(int)( oLoadedTexture.Size.X * oRenderSettings.ZoomLevel ),
						(int)( oLoadedTexture.Size.Y * oRenderSettings.ZoomLevel ) );
					Globals.Inst().GameEngine.RenderTexture2D( sTextureKey,
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( 0,0 ),0,oRenderSettings.PercentTransparent,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
				else
				{
					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - m_vSize.X / 2f ),
						(int)( vScreenPt.Y - m_vSize.Y / 2f ),
						(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
						(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );
					nAngle = DSMath.CalculateRadAngle( 0,0,this.Vel.X,this.Vel.Y );

					Globals.Inst().GameEngine.RenderTexture2D( "Plasma_Projectile",
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( m_vSize.X / 2f,m_vSize.Y / 2f ),/*base.Angle*/nAngle + Math.PI / 2.0,
						oRenderSettings.PercentTransparent,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
			}
		}
        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			if( oCollidedWith.OwnerSocketID != this.OwnerSocketID && oCollidedWith.EntityType != enumEntityType.Loot )
			{
				if( m_bExploding == true && m_nExplodeDamage > 0 )
				{
                    oCollidedWith.DealDamage(oSession, this, nElapsedTime * m_nExplodeDamage, oCollisionRegion);
				}
				else if( m_bExploding == false )
				{
					//if( m_nExplodeDamage > 0 )
					//{
						m_bExploding = true;
						m_nExplosionStartTime = base.TotalElapsedTime;
					//}

					this.RecoilsAfterCollision = false;
                    oCollidedWith.DealDamage(oSession, this, m_nImpactDamage, oCollisionRegion);
					base.Vel = new Vector2( 0,0 );				
				}

				return( true );
			}
			else
			{
				return( false );
			}
		}
		public override ArrayList GetRegions()
		{
			RegionCircle oCircle = null;
			ArrayList oRegions = new ArrayList();

			if( m_bExploding == true )
			{
				//Increase our damage radius if we are xploding
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y );
			}
			else
			{
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y / 2.0f );
			}

			//if( base.TotalElapsedTime > .25 )
		{
			oRegions.Add( oCircle );
		}

			return( oRegions );
		}


		private int GetExplosionFrame()
		{
			double nTotalExploTime = base.TotalElapsedTime - m_nExplosionStartTime;
			int nFrame = (int)Math.Floor( nTotalExploTime / .04 );
			if( nFrame < 0 )
			{
				nFrame = 0;
			}

			return( nFrame );
		}


		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nExplosionStartTime );
			oSerialize.Set( 2,m_bExploding );
			oSerialize.Set( 3,m_nImpactDamage );
			oSerialize.Set( 4,m_nExplodeDamage );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			//m_nExplosionStartTime = oSerialize.GetDouble( 1 );
			m_bExploding = oSerialize.GetBool( 2 );
			m_nImpactDamage = oSerialize.GetDouble( 3 );
			m_nExplodeDamage = oSerialize.GetDouble( 4 );
		}



		#region Properties
		public override long EXPValue
		{
			get
			{
				return( 0 );
			}
		}
		public double ImpactDamage
		{
			get
			{
				return( m_nImpactDamage );
			}
			set
			{
				m_nImpactDamage = value;
			}
		}
		public double ExplodeDamage
		{
			get
			{
				return( m_nExplodeDamage );
			}
			set
			{
				m_nExplodeDamage = value;
			}
		}
		#endregion
	}
	public class ScatterProjectile : Entity
	{
		#region Properties
		private const double m_cSECONDALIVE = 1.2;

		private double m_nExplosionStartTime = 0;
		private bool m_bExploding = false;
		private bool m_bHavePlayedExplodingSound = false;
		private bool m_bHavePlayedLaunchSound = false;
		private Vector2 m_vSize = new Vector2( 3,3 );
		#endregion


		public ScatterProjectile()
		{
			base.RecoilsAfterCollision = false;
			base.MaxVelocity = 0;
			base.EntityType = enumEntityType.Projectile;
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
			//double nAccel = 80;


			if( m_bExploding == true && m_nExplosionStartTime == 0 )
			{
				m_nExplosionStartTime = base.TotalElapsedTime;
			}
			if( m_bExploding == true && m_bHavePlayedExplodingSound == false && Globals.Inst().IAmTheServer == false )
			{
				m_bHavePlayedExplodingSound = true;
                Globals.PlaySound("Scatter_Explosion",this.Location);
			}
			if( m_bHavePlayedLaunchSound == false && Globals.Inst().IAmTheServer == false )
			{
				m_bHavePlayedLaunchSound = true;
                Globals.PlaySound("Scatter_Launch", this.Location);
			}

			if( base.TotalElapsedTime > m_cSECONDALIVE )
			{
				Globals.Inst().Session.RemoveEntity( this );
			}

			if( m_bExploding == false )
			{
                base.Advance(oSession, nElapsedTime);
			}
			else
			{
				//Advance our explosion
				base.TimeSinceLastLiteSend += nElapsedTime;
				base.TotalElapsedTime += nElapsedTime;
				base.TimeSinceLastCollision += nElapsedTime;
				base.TimeSinceLastFullSend += nElapsedTime;
			}

			if( m_bExploding == true )//&& GetExplosionFrame() > 19 )
			{
				Globals.Inst().Session.RemoveEntity( this );
			}
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			string sTextureKey = "";
			LoadedTexture oLoadedTexture = null;
			Vector2 vScreenPt = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,base.Pos );

				//Draw our ship
				if( m_bExploding == true && GetExplosionFrame() <= 19 )
				{
					sTextureKey = "Scatter_Explo_" + GetExplosionFrame().ToString();

					oLoadedTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sTextureKey );
					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - oLoadedTexture.Size.X / 2f ),
						(int)( vScreenPt.Y - oLoadedTexture.Size.Y / 2f ),
						(int)( oLoadedTexture.Size.X * oRenderSettings.ZoomLevel ),
						(int)( oLoadedTexture.Size.Y * oRenderSettings.ZoomLevel ) );
					Globals.Inst().GameEngine.RenderTexture2D( sTextureKey,
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( 0,0 ),0,oRenderSettings.PercentTransparent,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
				else
				{
					if( oRenderSettings.BaseDrawColor == System.Drawing.Color.Blue )
					{
						oRenderRect = new System.Drawing.Rectangle( 
							(int)( vScreenPt.X - m_vSize.X / 2f ),
							(int)( vScreenPt.Y - m_vSize.Y / 2f-25 ),
							(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
							(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );
					}
					else if( oRenderSettings.BaseDrawColor == System.Drawing.Color.Green )
					{
						oRenderRect = new System.Drawing.Rectangle( 
							(int)( vScreenPt.X - m_vSize.X / 2f ),
							(int)( vScreenPt.Y - m_vSize.Y / 2f-50 ),
							(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
							(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );
					}
					else
					{
						oRenderRect = new System.Drawing.Rectangle( 
							(int)( vScreenPt.X - m_vSize.X / 2f ),
							(int)( vScreenPt.Y - m_vSize.Y / 2f ),
							(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
							(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );
					}
					Globals.Inst().GameEngine.RenderTexture2D( "Scatter_Projectile",
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( m_vSize.X / 2f,m_vSize.Y / 2f ),base.Angle + Math.PI / 2.0,
						oRenderSettings.PercentTransparent,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
			}
		}
        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			if( m_bExploding == true )
			{
				//oCollidedWith.DealDamage( this,nElapsedTime * 5,oCollisionRegion );
			}
			else if( oCollidedWith.OwnerSocketID != this.OwnerSocketID )
			{
				this.RecoilsAfterCollision = false;
				m_bExploding = true;
				m_nExplosionStartTime = base.TotalElapsedTime;
                oCollidedWith.DealDamage(oSession, this, 2, oCollisionRegion);
				base.Vel = new Vector2( 0,0 );
			}

			return( true );
		}
		public override ArrayList GetRegions()
		{
			RegionCircle oCircle = null;
			ArrayList oRegions = new ArrayList();

			if( m_bExploding == true )
			{
				//Increase our damage radius if we are xploding
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y * 2 );
			}
			else
			{
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y );
			}

			//if( base.TotalElapsedTime > .25 )
		{
			oRegions.Add( oCircle );
		}

			return( oRegions );
		}

		public override double DelayBetweenFullSends()
		{
			return( 5 );
		}
		public override double DelayBetweenLiteSends()
		{
			return( 5 );
		}

		private int GetExplosionFrame()
		{
			int nFrame = 0;
			/*double nTotalExploTime = base.TotalElapsedTime - m_nExplosionStartTime;
			int nFrame = (int)Math.Floor( nTotalExploTime / .01 );
			if( nFrame < 0 )
			{
				nFrame = 0;
			}*/

			return( nFrame );
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nExplosionStartTime );
			oSerialize.Set( 2,m_bExploding );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			//m_nExplosionStartTime = oSerialize.GetDouble( 1 );
			m_bExploding = oSerialize.GetBool( 2 );
		}


		#region Properties
		public override long EXPValue
		{
			get
			{
				return( 0 );
			}
		}
		#endregion
	}	

	public class Projectile : Entity 
	{
		#region Properties
		//Sound Variables
		private string m_sLaunchSoundKey = string.Empty;
		private bool m_bHavePlayedLaunchSound = false;

		//Rendering variables
		private Vector2 m_vSize = new Vector2( GraphicConstants.m_cGRID_WIDTH,2 * GraphicConstants.m_cGRID_HEIGHT );
		private string m_sProjectileTexture = string.Empty;

		//Life variables
		private double m_nLifetimeInSeconds = 4;

		//Motion Variables

		private double m_nAcceleration = 0;
		private bool m_bHoming = false;
		private bool m_bAutoTarget = true;
		private Entity m_oHomingTarget = null;
		private double m_nTimeSinceLastHomingUpdate = 10;

		private double m_nImpactDamageMin = 0;
		private double m_nImpactDamageMax = 0;
		private double m_nExplodingDamage = 0;
		private bool m_bExploding = false;
		private double m_nExplosionStartTime = 0;
		private string m_sExplosionSpriteName = string.Empty;
		//private int m_nExplosionFrames = 0;
		private double m_nTimePerFrame = .03;
		#endregion


		public Projectile()
		{
			base.RecoilsAfterCollision = false;
			base.MaxVelocity = 0;
			base.EntityType = enumEntityType.Projectile;
		}

        public override void Advance(Session oSession,double nElapsedTime)
		{
			Entity oClosestTarget = null;
            Zone oMyZone = null;
			double nAccelAngle = 0;


			try
			{
                oMyZone = oSession.Zones[this.MostUpToDateLoc.ZoneID.ToString()];

				//Handle launch sounds
				if( m_sLaunchSoundKey.Length > 0 && Globals.Inst().IAmTheServer == false && m_bHavePlayedLaunchSound == false )
				{
					m_bHavePlayedLaunchSound = true;
                    Globals.PlaySound(m_sLaunchSoundKey, this.Location);
				}

				//Homing
				nAccelAngle = this.Angle;
				if( m_bHoming == true )
				{
					if( m_oHomingTarget != null )
					{
                        if( oMyZone.Entitys.ContainsKey( m_oHomingTarget.PKey ) == true ) 
						{
							//Accelerate towords the homing target
							nAccelAngle = DSMath.CalculateRadAngle( this.Pos.X,this.Pos.Y,m_oHomingTarget.Pos.X,m_oHomingTarget.Pos.Y );
						}
						else
						{
							m_oHomingTarget = null;
						}
					}
						
					m_nTimeSinceLastHomingUpdate += nElapsedTime;
					if( m_bAutoTarget == true && m_nTimeSinceLastHomingUpdate > 1 )
					{
						m_nTimeSinceLastHomingUpdate = 0;
						//Did we find a target?
						oClosestTarget = FindClosestTarget(oSession);
						if( oClosestTarget != null &&
							DSMisc.Distance( oClosestTarget.Pos.X,oClosestTarget.Pos.Y,this.Pos.X,this.Pos.Y ) < 4000 )
						{
							this.HomingTarget = oClosestTarget;
						}
						else
						{
							this.HomingTarget = null;
						}
					}
				}

				//Motion
				if( m_nAcceleration != 0 )
				{
					//If we are slowing down, and we just started going in reverse, stop us.
					if( m_nAcceleration < 0 && this.Vel.Length() < m_nAcceleration * nElapsedTime  )
					{
						this.Vel = new Vector2( 0,0 );
					}
					else
					{
						this.Vel = new Vector2( 
							(float)( this.Vel.X + Math.Cos( nAccelAngle ) * m_nAcceleration * nElapsedTime ),
							(float)( this.Vel.Y + Math.Sin( nAccelAngle ) * m_nAcceleration * nElapsedTime ) );

					}
				}

				//Have we exceeded our lifetime?
				if( base.TotalElapsedTime > m_nLifetimeInSeconds && m_nLifetimeInSeconds > 0  )
				{
					this.IsDead = true;
                    oSession.RemoveEntity(this);
				}

                base.Advance(oSession, nElapsedTime);
			}
			catch( System.Exception oEx )
			{
				DSMisc.ShowErrors( oEx );
			}
		}
		public override void Render( RenderSettings oRenderSettings )
		{
			string sTextureKey = string.Empty;
			double nPercTrans = 0;
			double nAngle = 0;
			LoadedTexture oLoadedTexture = null;
			Vector2 vScreenPt = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.AboveShip && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				//Draw our ship
				if( m_bExploding == true && GetExplosionFrame() <= 9 )
				{
					sTextureKey = m_sExplosionSpriteName + GetExplosionFrame().ToString();

					oLoadedTexture = DSResourceManager.GetGlobalInstance().GetLoadedTexture( sTextureKey );
					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - oLoadedTexture.Size.X ),
						(int)( vScreenPt.Y - oLoadedTexture.Size.Y ),
						(int)( oLoadedTexture.Size.X * 2 * oRenderSettings.ZoomLevel ),
						(int)( oLoadedTexture.Size.Y * 2 * oRenderSettings.ZoomLevel ) );
					Globals.Inst().GameEngine.RenderTexture2D( sTextureKey,
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( 0,0 ),0,oRenderSettings.PercentTransparent,
						false,oRenderSettings.BaseDrawColor.ToArgb() );
				}
				else
				{
					vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,base.Pos );

					oRenderRect = new System.Drawing.Rectangle( 
						(int)( vScreenPt.X - m_vSize.X / 2f ),
						(int)( vScreenPt.Y - m_vSize.Y / 2f ),
						(int)( m_vSize.X * oRenderSettings.ZoomLevel ),
						(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );
					nAngle = DSMath.CalculateRadAngle( 0,0,this.Vel.X,this.Vel.Y );

					nPercTrans = DSMisc.Max( oRenderSettings.PercentTransparent,this.FadeoutTransparency() );

					Globals.Inst().GameEngine.RenderTexture2D( m_sProjectileTexture,
						System.Drawing.Rectangle.Empty,oRenderRect,
						new Vector2( oRenderRect.Width / 2f,oRenderRect.Height / 2f ),
						nAngle + Math.PI / 2.0,
						nPercTrans,false,
						oRenderSettings.BaseDrawColor.ToArgb() );
				}
			}
		}
		private int GetExplosionFrame()
		{
			double nTotalExploTime = base.TotalElapsedTime - m_nExplosionStartTime;
			int nFrame = (int)Math.Floor( nTotalExploTime / m_nTimePerFrame );
			if( nFrame < 0 )
			{
				nFrame = 0;
			}

			return( nFrame );
		}

		private Entity FindClosestTarget(Session oSession)
		{
			double nTempDist = 0;
			double nDistToEntity = 0;
			Entity oClosestEntity = null;


            foreach( Entity oLoopEntity in oSession.GetEntitysInMyZone(this).Values )
			{
				nTempDist = DSMath.Distance( this.Pos.X,this.Pos.Y,oLoopEntity.Pos.X,oLoopEntity.Pos.Y );
				if( oLoopEntity.OwnerSocketID != this.OwnerSocketID &&
					oLoopEntity.EntityType == enumEntityType.Ship &&
                    oLoopEntity.IsDead == false &&
					( nTempDist < nDistToEntity || oClosestEntity == null ) )
				{
					nDistToEntity = nTempDist;
					oClosestEntity = oLoopEntity;
				}
			}


			return( oClosestEntity );
		}

        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
			bool bWasDeadBefore = this.IsDead;
			double nDamageRemaining = 0;

            nDamageRemaining = base.DealDamage(oSession, oEntityDealingDamage, nAmount, oDamageWhere);
			if( this.IsDead == true && bWasDeadBefore == false )
			{
                oSession.RemoveEntity(this);
			}

			return( nDamageRemaining );
		}
		public override ArrayList GetRegions()
		{
			RegionCircle oCircle = null;
			ArrayList oRegions = new ArrayList();

			oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,m_vSize.Y );
			oRegions.Add( oCircle );

			return( oRegions );
		}
        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			double nDamage = 0;


			if( this.IsDead == false && oCollidedWith.OwnerSocketID != this.OwnerSocketID )
			{
				base.Vel = new Vector2( 0,0 );

				if( m_bExploding == true )
				{
                    oCollidedWith.DealDamage(oSession, this, nElapsedTime * m_nExplodingDamage, oCollisionRegion);
				}
				else if( m_nExplodingDamage > 0 )
				{
					this.RecoilsAfterCollision = false;
					m_bExploding = true;
					m_nExplosionStartTime = base.TotalElapsedTime;
					nDamage = DSMisc.GetRnd( m_nImpactDamageMin,m_nImpactDamageMax );
                    oCollidedWith.DealDamage(oSession, this, nDamage, oCollisionRegion);
				}
				else
				{
					nDamage = DSMisc.GetRnd( m_nImpactDamageMin,m_nImpactDamageMax );
                    oCollidedWith.DealDamage(oSession, this, nDamage, oCollisionRegion);
					this.IsDead = true;
                    oSession.RemoveEntity(this);
				}

				return( true );
			}
			else
			{
				return( false );
			}
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			oSerialize.Set( 1,m_nImpactDamageMin );
			oSerialize.Set( 2,m_nImpactDamageMax );
			oSerialize.Set( 3,m_nExplodingDamage );
			oSerialize.Set( 4,m_bExploding );
			oSerialize.Set( 5,m_nExplosionStartTime );


			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 0 ) );
			m_nImpactDamageMin = oSerialize.GetDouble( 1 );
			m_nImpactDamageMax = oSerialize.GetDouble( 2 );
			m_nExplodingDamage = oSerialize.GetDouble( 3 );
			m_bExploding = oSerialize.GetBool( 4 );
			m_nExplosionStartTime = oSerialize.GetDouble( 5 );
		}


		#region Properties
		public override long EXPValue
		{
			get
			{
				return( 0 );
			}
		}
		public string LaunchSoundKey
		{
			get
			{
				return( m_sLaunchSoundKey );
			}
			set
			{
				m_sLaunchSoundKey = value;
			}
		}
		public bool Homing
		{
			get
			{
				return( m_bHoming );
			}
			set
			{
				m_bHoming = value;
			}
		}
		public double Acceleration
		{
			get
			{
				return( m_nAcceleration );
			}
			set
			{
				m_nAcceleration = value;
			}
		}
		public Entity HomingTarget
		{
			get
			{
				return( m_oHomingTarget );
			}
			set
			{
				m_oHomingTarget = value;
			}
		}
		public Vector2 Size
		{
			get
			{
				return( m_vSize );
			}
			set
			{
				m_vSize = value;
			}
		}
		public string ProjectileTexture
		{
			get
			{
				return( m_sProjectileTexture );
			}
			set
			{
				m_sProjectileTexture = value;
			}
		}
		public double LifetimeInSeconds
		{
			get
			{
				return( m_nLifetimeInSeconds );
			}
			set
			{
				m_nLifetimeInSeconds = value;
			}
		}
		public double ImpactDamageMin
		{
			get
			{
				return( m_nImpactDamageMin );
			}
			set
			{
				m_nImpactDamageMin = value;
			}
		}
		public double ImpactDamageMax
		{
			get
			{
				return( m_nImpactDamageMax );
			}
			set
			{
				m_nImpactDamageMax = value;
			}
		}
		public double ExplodingDamage
		{
			get
			{
				return( m_nExplodingDamage );
			}
			set
			{
				m_nExplodingDamage = value;
			}
		}
		public bool AutoTarget
		{
			get
			{
				return( m_bAutoTarget );
			}
			set
			{
				m_bAutoTarget = value;
			}
		}
		#endregion
	}
	public class LimpitProjectile : Projectile
	{
		#region Properties
		private const double m_cATTACHTIME = 60;

		private double m_nTimeSinceLastAIUpdate = 200;
		private double m_nTimeOfAttach = 0;
		private Entity m_oEntityAttachedTo = null;
		private Vector2 m_vAttachLocation = Vector2.Empty;
		#endregion


		public LimpitProjectile()
		{
			base.Size = new Vector2( GraphicConstants.m_cGRID_WIDTH,GraphicConstants.m_cGRID_HEIGHT );
			base.LaunchSoundKey = "Vux_Limpit_Launch";
			base.ProjectileTexture = "Limpit_Projectile";
			base.RecoilsAfterCollision = false;
			base.Acceleration = 150;
			base.MaxVelocity = 150;
			base.LifetimeInSeconds = 20;
			base.Homing = true;
		}


		public override void Render(RenderSettings oRenderSettings)
		{
			if( m_oEntityAttachedTo != null )
			{
				this.Pos = m_oEntityAttachedTo.GetRotatedWorldPt( 
					new Vector2( m_oEntityAttachedTo.Pos.X + m_vAttachLocation.X,
					m_oEntityAttachedTo.Pos.Y + m_vAttachLocation.Y ) );
			}
			base.Render (oRenderSettings);
		}

        public override void Advance(Session oSession, double nElapsedTime)
		{
            Zone oMyZone = null;


            base.Advance(oSession, nElapsedTime);
            oMyZone = oSession.Zones[this.MostUpToDateLoc.ZoneID.ToString()];

			if( m_oEntityAttachedTo == null )
			{
				m_nTimeSinceLastAIUpdate += nElapsedTime;
				if( m_nTimeSinceLastAIUpdate > 15 )
				{
					//Did we find a target?
					this.HomingTarget = FindClosestShip(oSession);
				}
			}
            else if (oSession.GetEntitysInMyZone(this).ContainsKey(m_oEntityAttachedTo.PKey) == true )
			{
                oSession.RemoveEntity(this);
			}
			else if( Globals.Inst().GameEngine.AppTime - m_nTimeOfAttach > m_cATTACHTIME )
			{
				this.StartFadeout( 2 );
			}
		}
		private Entity FindClosestShip(Session oSession)
		{
			double nTempDist = 0;
			double nDistToEntity = 0;
    		Entity oClosestEntity = null;


            foreach (Entity oLoopEntity in oSession.GetEntitysInMyZone(this).Values)
			{
				nTempDist = DSMath.Distance( this.Pos.X,this.Pos.Y,oLoopEntity.Pos.X,oLoopEntity.Pos.Y );
				if( oLoopEntity.OwnerSocketID != this.OwnerSocketID &&
					oLoopEntity.EntityType == enumEntityType.Ship &&
					( nTempDist < nDistToEntity || oClosestEntity == null ) )
				{
					nDistToEntity = nTempDist;
					oClosestEntity = oLoopEntity;
				}
			}


			return( oClosestEntity );
		}
        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
		{
			int nRegionIdx = 0;
			double nAngle = 0;
			double nDist = 0;
			RegionCircle oRegionCircle = null;
			ArrayList oRegions = null;


			if( m_oEntityAttachedTo == null && oCollidedWith.OwnerSocketID != this.OwnerSocketID )
			{
				if( oCollidedWith.EntityType != enumEntityType.Loot && oCollidedWith.EntityType != enumEntityType.Projectile )
				{
					m_oEntityAttachedTo = oCollidedWith;
					m_nTimeOfAttach = Globals.Inst().GameEngine.AppTime;
					base.LifetimeInSeconds = 0;

					//Stop our rotation and motion
					this.AngularMomentum = 0;
					base.Homing = false;
					base.Acceleration = 0;

					//Assign penalties
					m_oEntityAttachedTo.AddTemporaryProperty( enumEntProperties.Movement_EtherialLinearDrag,-30,m_cATTACHTIME );
					m_oEntityAttachedTo.AddTemporaryProperty( enumEntProperties.Movement_EtherialAngularDrag,-2,m_cATTACHTIME );

					//Pick an attach location
					oRegions = m_oEntityAttachedTo.GetRegions();
					nRegionIdx = DSMisc.GetRnd( 0,oRegions.Count - 1 );
					oRegionCircle = (RegionCircle)oRegions[ nRegionIdx ];
					nAngle = DSMisc.GetRnd() * Math.PI * 2;
					nDist = DSMisc.GetRnd() * oRegionCircle.Radius;
					m_vAttachLocation = new Vector2( (float)( Math.Cos( nAngle ) * nDist ),
						(float)( Math.Sin( nAngle ) * nDist ) );

					this.CollidesWithOtherEntitys = false;

					if( Globals.Inst().IAmTheServer == false )
					{
                        Globals.PlaySound("Vux_Limpit_Hit", this.Location);
					}
				}		
				else
				{
                    oSession.RemoveEntity(this);
				}

				return( true );
			}
			else
			{
				return( false );
			}			
		}

		public override ArrayList GetRegions()
		{
			RegionCircle oCircle = null;
			ArrayList oRegions = new ArrayList();

			if( m_oEntityAttachedTo == null )
			{
				oCircle = new RegionCircle( base.MostUpToDateLoc.Pos,this.Size.Y );
				oRegions.Add( oCircle );
			}

			return( oRegions );
		}


		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
			if( m_oEntityAttachedTo != null )
			{
				oSerialize.Set( 1,m_oEntityAttachedTo.PKeyObject.Serialize() );
			}


			return( oSerialize );
		}
		public override void DeSerialize( Session oSession,DSSerialize oSerialize )
		{
            Zone oMyZone = null;
			object oObject = null;
			DSNetPKey oPKey = null;
			string sPKey = string.Empty;
            Location oLoc = null;


            base.DeSerialize(oSession, (DSSerialize)oSerialize.Get(0));

            oLoc = this.MostUpToDateLoc;
            oMyZone = oSession.Zones[oLoc.ZoneID.ToString()];

			//Get the key if there is one
			oObject = oSerialize.Get( 1 );
			if( oObject != null )
			{
				oPKey = new DSNetPKey( 0,0 );
				oPKey.DeSerialize( (DSSerialize)oSerialize.Get( 1 ) );
                m_oEntityAttachedTo = oSession.GetEntity(oPKey.ToString());
			}
		}



		#region Properties
		#endregion
	}
	public class PlasmaPellitProjectile : Projectile
	{
		public PlasmaPellitProjectile()
		{
			base.Size = new Vector2( GraphicConstants.m_cGRID_WIDTH,GraphicConstants.m_cGRID_HEIGHT );
			base.LaunchSoundKey = "Spathi_Plasma_Launch";
			base.ProjectileTexture = "Spathi_Pellet_Projectile";
			base.RecoilsAfterCollision = false;
			base.LifetimeInSeconds = 2;
		}
	}
	public class BUTTProjectile : Projectile
	{
		public BUTTProjectile()
		{
			base.Size = new Vector2( GraphicConstants.m_cGRID_WIDTH,GraphicConstants.m_cGRID_HEIGHT );
			base.LaunchSoundKey = "Spathi_BUTT_Launch";
			base.ProjectileTexture = "Spathi_BUTT_Projectile";
			base.RecoilsAfterCollision = false;
			base.LifetimeInSeconds = 12;
			base.Homing = true;
			base.AutoTarget = true;
			base.Acceleration = 200;
			base.MaxVelocity = 150;
		}
	}
}
